<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kale Miner Dashboard</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header {
            background: white;
            padding: 20px 30px;
            border-radius: 15px;
            margin-bottom: 20px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            color: #667eea;
            font-size: 28px;
        }
        
        .refresh-info {
            font-size: 14px;
            color: #666;
        }
        
        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }
        
        .tab-btn {
            background: white;
            border: none;
            padding: 12px 24px;
            border-radius: 10px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
            transition: all 0.3s;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .tab-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        
        .tab-btn.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .server-panel {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        
        .server-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 20px;
            border-bottom: 2px solid #f0f0f0;
        }
        
        .server-name {
            font-size: 24px;
            font-weight: 700;
            color: #333;
        }
        
        .status-badge {
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }
        
        .status-online {
            background: #10b981;
            color: white;
        }
        
        .status-warning {
            background: #f59e0b;
            color: white;
        }
        
        .status-offline {
            background: #ef4444;
            color: white;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 12px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        
        .stat-label {
            font-size: 14px;
            opacity: 0.9;
            margin-bottom: 8px;
        }
        
        .stat-value {
            font-size: 32px;
            font-weight: 700;
        }
        
        .stat-sub {
            font-size: 12px;
            opacity: 0.8;
            margin-top: 5px;
        }
        
        .section-title {
            font-size: 18px;
            font-weight: 600;
            margin: 25px 0 15px 0;
            color: #333;
        }
        
        .tmux-status {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(80px, 1fr));
            gap: 8px;
            margin-bottom: 30px;
        }
        
        .tmux-item {
            padding: 10px;
            border-radius: 8px;
            text-align: center;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s;
        }
        
        .tmux-item:hover {
            transform: scale(1.05);
        }
        
        .tmux-online {
            background: #d1fae5;
            color: #065f46;
        }
        
        .tmux-offline {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .tokens-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 10px;
        }
        
        .tokens-table th {
            background: #f9fafb;
            padding: 12px;
            text-align: left;
            font-weight: 600;
            color: #374151;
            border-bottom: 2px solid #e5e7eb;
        }
        
        .tokens-table td {
            padding: 12px;
            border-bottom: 1px solid #e5e7eb;
        }
        
        .tokens-table tr:hover {
            background: #f9fafb;
        }
        
        .token-value {
            font-family: monospace;
            font-size: 11px;
            color: #6b7280;
            max-width: 400px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        
        .token-credits {
            font-weight: 600;
            color: #10b981;
        }
        
        .token-expiry {
            font-size: 12px;
            color: #9ca3af;
        }
        
        .btn-detail {
            background: #667eea;
            color: white;
            border: none;
            padding: 6px 12px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 12px;
            transition: all 0.3s;
        }
        
        .btn-detail:hover {
            background: #764ba2;
        }
        
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            backdrop-filter: blur(4px);
        }
        
        .modal-content {
            background: white;
            margin: 5% auto;
            padding: 30px;
            border-radius: 15px;
            width: 90%;
            max-width: 600px;
            box-shadow: 0 10px 25px rgba(0,0,0,0.3);
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f0f0f0;
        }
        
        .close {
            font-size: 28px;
            font-weight: bold;
            color: #aaa;
            cursor: pointer;
            transition: color 0.3s;
        }
        
        .close:hover {
            color: #000;
        }
        
        .token-full {
            background: #f9fafb;
            padding: 15px;
            border-radius: 8px;
            word-break: break-all;
            font-family: monospace;
            font-size: 12px;
            line-height: 1.6;
            max-height: 300px;
            overflow-y: auto;
        }
        
        .loading {
            text-align: center;
            padding: 40px;
            color: #9ca3af;
        }
        
        .error {
            background: #fee2e2;
            color: #991b1b;
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
        }
        
        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .tmux-status {
                grid-template-columns: repeat(auto-fill, minmax(60px, 1fr));
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🌾 Kale Miner Dashboard</h1>
            <div class="refresh-info">
                <span id="lastUpdate">Loading...</span><br>
                <small>Auto-refresh: 30s</small>
            </div>
        </div>
        
        <div class="tabs" id="serverTabs"></div>
        
        <div id="serverContent"></div>
    </div>
    
    <!-- Modal for token detail -->
    <div id="tokenModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="modalTitle">Token Detail</h3>
                <span class="close">&times;</span>
            </div>
            <div id="modalBody"></div>
        </div>
    </div>
    
    <script>
        const SERVERS = [
            { id: 'server_putih', name: 'Server Putih', tokens: 20, configs: 200 },
            { id: 'server_fajar', name: 'Server Fajar', tokens: 20, configs: 200 },
            { id: 'server_obet', name: 'Server Obet', tokens: 10, configs: 100 }
        ];
        
        const BASE_URL = window.location.origin + window.location.pathname.replace('dashboard.php', '');
        let currentServer = SERVERS[0].id;
        
        // Initialize
        function init() {
            renderTabs();
            loadAllData();
            setInterval(loadAllData, 30000); // Auto-refresh every 30s
        }
        
        // Render server tabs
        function renderTabs() {
            const tabsContainer = document.getElementById('serverTabs');
            tabsContainer.innerHTML = SERVERS.map(server => 
                `<button class="tab-btn ${server.id === currentServer ? 'active' : ''}" 
                         onclick="switchServer('${server.id}')">
                    ${server.name}
                </button>`
            ).join('');
        }
        
        // Switch server
        function switchServer(serverId) {
            currentServer = serverId;
            renderTabs();
            renderServerContent();
        }
        
        // Load all data
        async function loadAllData() {
            const results = await Promise.all(SERVERS.map(async server => {
                try {
                    const [credits, tokens, tmux, tokenCredits, wallet] = await Promise.all([
                        fetch(`${BASE_URL}data/credits/${server.id}.json`).then(r => r.ok ? r.json() : null),
                        fetch(`${BASE_URL}data/tokens/${server.id}.json`).then(r => r.ok ? r.json() : null),
                        fetch(`${BASE_URL}data/tmux/${server.id}.json`).then(r => r.ok ? r.json() : null),
                        fetch(`${BASE_URL}data/token-credits/${server.id}.json`).then(r => r.ok ? r.json() : null),
                        fetch(`${BASE_URL}data/wallet/${server.id}.json`).then(r => r.ok ? r.json() : null)
                    ]);
                    return { server, credits, tokens, tmux, tokenCredits, wallet, error: null };
                } catch (error) {
                    return { server, credits: null, tokens: null, tmux: null, tokenCredits: null, wallet: null, error: error.message };
                }
            }));
            
            window.serverData = results.reduce((acc, item) => {
                acc[item.server.id] = item;
                return acc;
            }, {});
            
            renderServerContent();
            document.getElementById('lastUpdate').textContent = `Last update: ${new Date().toLocaleTimeString()}`;
        }
        
        // Render server content
        function renderServerContent() {
            const container = document.getElementById('serverContent');
            const data = window.serverData?.[currentServer];
            
            if (!data) {
                container.innerHTML = '<div class="loading">Loading data...</div>';
                return;
            }
            
            if (data.error) {
                container.innerHTML = `
                    <div class="server-panel">
                        <div class="error">❌ Error loading data: ${data.error}</div>
                    </div>
                `;
                return;
            }
            
            const { credits, tokens, tmux, tokenCredits, wallet } = data;
            const serverInfo = SERVERS.find(s => s.id === currentServer);
            const serverName = serverInfo?.name || currentServer;
            const maxTokens = serverInfo?.tokens || 20;
            const maxConfigs = serverInfo?.configs || 250;
            
            // Calculate tmux stats
            const tmuxOnline = tmux?.sessions?.filter(s => s.status === 'online').length || 0;
            const tmuxTotal = maxConfigs; // Use expected max configs for the server
            const tmuxOffline = tmuxTotal - tmuxOnline;
            const tmuxPercent = ((tmuxOnline / tmuxTotal) * 100).toFixed(1);
            
            // Get status
            let statusClass = 'status-online';
            let statusText = 'All Systems Operational';
            if (tmuxPercent < 50) {
                statusClass = 'status-offline';
                statusText = 'Critical';
            } else if (tmuxPercent < 90) {
                statusClass = 'status-warning';
                statusText = 'Warning';
            }
            
            // Get today's tokens with real-time credits
            const today = new Date().toISOString().split('T')[0];
            const todayTokens = tokenCredits?.dates?.[today]?.tokens || [];
            
            // Calculate token percentage
            const initialTokens = 150 * maxTokens; // 250 XLM per token * number of tokens
            const currentTokens = todayTokens.reduce((sum, t) => sum + (t.creditsXLM || 0), 0);
            const tokenPercent = ((currentTokens / initialTokens) * 100).toFixed(1);
            const tokenUsed = initialTokens - currentTokens;
            
            container.innerHTML = `
                <div class="server-panel">
                    <div class="server-header">
                        <div class="server-name">${serverName}</div>
                        <div class="status-badge ${statusClass}">${statusText}</div>
                    </div>
                    
                    <div class="stats-grid">
                        <div class="stat-card">
                            <div class="stat-label">Tmux Workers</div>
                            <div class="stat-value">${tmuxOnline}/${tmuxTotal}</div>
                            <div class="stat-sub">${tmuxPercent}% Online</div>
                        </div>
                        
                        <div class="stat-card">
                            <div class="stat-label">Total Wallet Balance</div>
                            <div class="stat-value">${wallet?.totalBalance?.toFixed(2) || '0'}</div>
                            <div class="stat-sub">KALE from ${wallet?.walletCount || 0} wallets</div>
                        </div>
                        
                        <div class="stat-card">
                            <div class="stat-label">Tokens Remaining</div>
                            <div class="stat-value">${tokenPercent}%</div>
                            <div class="stat-sub">${currentTokens.toFixed(2)}/${initialTokens} XLM (used: ${tokenUsed.toFixed(2)})</div>
                        </div>
                        
                        <div class="stat-card">
                            <div class="stat-label">Server Capacity</div>
                            <div class="stat-value">${maxTokens}T/${maxConfigs}W</div>
                            <div class="stat-sub">${maxTokens} Tokens, ${maxConfigs} Workers</div>
                        </div>
                        
                        <div class="stat-card">
                            <div class="stat-label">LaunchTube Credits</div>
                            <div class="stat-value">${credits?.creditsXLM?.toFixed(2) || '0'}</div>
                            <div class="stat-sub">XLM Remaining</div>
                        </div>
                    </div>
                    
                    <div class="section-title">Daily Tokens (${today})</div>
                    ${renderTokensTable(todayTokens)}
                </div>
            `;
        }
        
        // Render tokens table
        function renderTokensTable(tokens) {
            if (!tokens.length) {
                return '<p class="loading">No tokens data available</p>';
            }
            
            return `
                <table class="tokens-table">
                    <thead>
                        <tr>
                            <th>Token #</th>
                            <th>Credits Remaining</th>
                            <th>Expiry Date</th>
                            <th>Status</th>
                            
                        </tr>
                    </thead>
                    <tbody>
                        ${tokens.map(t => {
                            const decoded = decodeJWT(t.value);
                            const creditsXLM = t.creditsXLM !== undefined ? t.creditsXLM.toFixed(2) + ' XLM' : 'Checking...';
                            const statusClass = t.error ? 'status-offline' : (t.creditsXLM > 0 ? 'status-online' : 'status-warning');
                            const statusText = t.error ? 'Error' : (t.creditsXLM > 0 ? 'Active' : 'Empty');
                            
                            return `
                                <tr>
                                    <td><strong>Token ${t.idx}</strong></td>
                                    <td><span class="token-credits">${creditsXLM}</span></td>
                                    <td><span class="token-expiry">${decoded.expiry || 'N/A'}</span></td>
                                    <td><span class="status-badge ${statusClass}" style="font-size:10px;padding:4px 8px;">${statusText}</span></td>
                                    
                                </tr>
                            `;
                        }).join('')}
                    </tbody>
                </table>
            `;
        }
        
        // Decode JWT (simple base64 decode)
        function decodeJWT(token) {
            if (!token || token === '') return { creditsXLM: 'Empty', expiry: 'N/A' };
            
            try {
                const parts = token.split('.');
                if (parts.length !== 3) return { creditsXLM: 'Invalid', expiry: 'Invalid' };
                
                const payload = JSON.parse(atob(parts[1]));
                const creditsRaw = payload.credits || 0;
                const creditsXLM = (creditsRaw / 10000000).toFixed(2) + ' XLM';
                const expiry = payload.exp ? new Date(payload.exp * 1000).toLocaleString() : 'N/A';
                
                return { creditsXLM, creditsRaw, expiry, payload };
            } catch (e) {
                return { creditsXLM: 'Invalid', expiry: 'Invalid' };
            }
        }
        
        // Show token detail modal
        function showTokenDetail(token) {
            const modal = document.getElementById('tokenModal');
            const modalTitle = document.getElementById('modalTitle');
            const modalBody = document.getElementById('modalBody');
            
            const decoded = decodeJWT(token.value);
            
            modalTitle.textContent = `Token ${token.idx} Details`;
            modalBody.innerHTML = `
                <h4>JWT Payload:</h4>
                <div class="token-full">${JSON.stringify(decoded.payload || {}, null, 2)}</div>
                <br>
                <h4>Full Token:</h4>
                <div class="token-full">${token.value || 'Empty'}</div>
            `;
            
            modal.style.display = 'block';
        }
        
        // Close modal
        document.addEventListener('DOMContentLoaded', () => {
            const modal = document.getElementById('tokenModal');
            const close = document.querySelector('.close');
            
            close.onclick = () => modal.style.display = 'none';
            window.onclick = (e) => {
                if (e.target === modal) modal.style.display = 'none';
            };
            
            init();
        });
    </script>
</body>
</html>
d